/*
	Implementation of the Azimuth discrimination and resynthesis sound source
	separation algorithm by Dan Barry, Bob Lawlor and Eugene Coyle.

	This C code implementation (c) 2006 Avelino Herrera Morales

	Go to http://www.dmc.dit.ie/2002/research_ditme/dnbarry/ for further info
	about the algorithm.

	(c) 2006  Avelino Herrera Morales
	avelinoherrera@hotpop.com
	http://avelino.gabiot.com/blog

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/soundcard.h>
#include <sys/ioctl.h>
#include "fft.h"


#define  SAMPLE_RATE     44100

/*
#define  BUFFER_SIZE     256
#define  L2_BUFFER_SIZE  8
#define  STEP_SIZE       32
*/

#define  BUFFER_SIZE     4096
#define  L2_BUFFER_SIZE  12
#define  STEP_SIZE       1024


#define  AZIMUTH_SIZE            100
#define  AZIMUTH_POSITION        98   /* aproximate sax position */
#define  AZIMUTH_SUBSPACE_WIDTH  2
#define  AZIMUTH_POS_FROM        (AZIMUTH_POSITION - (AZIMUTH_SUBSPACE_WIDTH / 2))
#define  AZIMUTH_POS_TO          (AZIMUTH_POSITION + (AZIMUTH_SUBSPACE_WIDTH / 2))

#define  OUTPUT_GAIN  0.1

complex left_channel[BUFFER_SIZE];
complex right_channel[BUFFER_SIZE];
float left_freq_azimuth_plane[BUFFER_SIZE][AZIMUTH_SIZE];
float right_freq_azimuth_plane[BUFFER_SIZE][AZIMUTH_SIZE];
float left_resynthesis_peaks[BUFFER_SIZE];
float right_resynthesis_peaks[BUFFER_SIZE];
complex left_channel_resynthesis[BUFFER_SIZE];
complex right_channel_resynthesis[BUFFER_SIZE];
float hanning_window[BUFFER_SIZE];


/******************************************************************************
	Record buffers from fd.
*/
int buffer_rec(int fd, float *left, float *right, float *window) {
	static int step_offset = 0;
	int i, readed;
	short int temp_buffer[BUFFER_SIZE * 2];

	lseek(fd, step_offset * 2 * sizeof(short int), SEEK_SET);
	readed = read(fd, temp_buffer, BUFFER_SIZE * 2 * sizeof(short int));
	for (i = 0; i < BUFFER_SIZE; i++) {
		left[i] = (((float) temp_buffer[2 * i]) / 32768.0) * window[i];
		right[i] = (((float) temp_buffer[(2 * i) + 1]) / 32768.0) * window[i];
	}
	step_offset += STEP_SIZE;
	if (readed < (BUFFER_SIZE * 2 * sizeof(short int)))
		return 0;
	return 1;
}

/******************************************************************************
	Play buffers throw fd.
*/
void buffer_play(int fd, float *left, float *right, float *window) {
	int i;
	int trash;
	static float left_mix_buffer[STEP_SIZE + BUFFER_SIZE];
	static float right_mix_buffer[STEP_SIZE + BUFFER_SIZE];
	short int temp_buffer[STEP_SIZE * 2];

	/* play the first STEP_SIZE samples of the mix buffers */
	for (i = 0; i < STEP_SIZE; i++) {
		register float l, r;
		l = left_mix_buffer[i] * OUTPUT_GAIN;
		if (l > 1.0)
			l = 1.0;
		else if (l < -1.0)
			l = -1.0;
		r = right_mix_buffer[i] * OUTPUT_GAIN;
		if (r > 1.0)
			r = 1.0;
		else if (r < -1.0)
			r = -1.0;
		temp_buffer[2 * i] = (short) rint(l * 32767);
		temp_buffer[(2 * i) + 1] = (short) rint(r * 32767);
	}
	write(fd, temp_buffer, STEP_SIZE * 2 * sizeof(short int));
	/* shift left the mix buffers STEP_SIZE samples */
	memmove(left_mix_buffer, &(left_mix_buffer[STEP_SIZE]), sizeof(float) * BUFFER_SIZE);
	memmove(right_mix_buffer, &(right_mix_buffer[STEP_SIZE]), sizeof(float) * BUFFER_SIZE);
	bzero(&(left_mix_buffer[BUFFER_SIZE]), STEP_SIZE);
	bzero(&(right_mix_buffer[BUFFER_SIZE]), STEP_SIZE);
	/* mix the inputs (left and right) on the rest of mix buffers */
	for (i = 0; i < BUFFER_SIZE; i++) {
		/*
		float v, w1, w2;
		if ((i + STEP_SIZE) >= BUFFER_SIZE)
			w1 = 0;
		else
			w1 = window[i + STEP_SIZE];
		w2 = window[i];
		v = (left_mix_buffer[i] * w1) + (left[i] * w2);
		left_mix_buffer[i] = v / (w1 + w2);
		v = (right_mix_buffer[i] * w1) + (right[i] * w2);
		right_mix_buffer[i] = v / (w1 + w2);
		*/
		left_mix_buffer[i] += left[i];
		right_mix_buffer[i] += right[i];
	}
}

/******************************************************************************
	Open /dev/dsp for listen.
*/
int audio_open(void) {
	int rate = SAMPLE_RATE;
	int format = AFMT_S16_LE;
	int stereo = 2;
	int fd;

	if ((fd = open("/dev/dsp", O_WRONLY, 0)) == -1) {
		perror("/dev/dsp");
		exit(-1);
	}
	ioctl(fd, SNDCTL_DSP_SPEED, &rate);
	ioctl(fd, SNDCTL_DSP_SETFMT, &format);
	ioctl(fd, SNDCTL_DSP_STEREO, &stereo);
	return fd;
}

/*****************************************************************************/
int main(int argc, char *argv[]) {
	int i, j;
	float left_input[BUFFER_SIZE], right_input[BUFFER_SIZE];
	float left_output[BUFFER_SIZE], right_output[BUFFER_SIZE];

	fill_hanning_window(hanning_window, BUFFER_SIZE);
	while (buffer_rec(0, left_input, right_input, hanning_window)) {
		/* calculate FFT over left and right channels */
		for (i = 0; i < BUFFER_SIZE; i++) {
			left_channel[i].re = left_input[i];
			left_channel[i].im = 0.0;
			right_channel[i].re = right_input[i];
			right_channel[i].im = 0.0;
		}
		fft(left_channel, L2_BUFFER_SIZE);
		fft(right_channel, L2_BUFFER_SIZE);
		/* TEST FFT
		ifft(left_channel, L2_BUFFER_SIZE);
		ifft(right_channel, L2_BUFFER_SIZE);
		for (i = 0; i < BUFFER_SIZE; i++) {
			left_output[i] = left_channel[i].re;
			right_output[i] = right_channel[i].re;
		}
		*/
		/* generate the frequency-azimuth plane for each channel */
		for (i = 0; i < BUFFER_SIZE; i++) {
			complex left_aux2, right_aux2;
			float left_abs, right_abs;
			for (j = 0; j < AZIMUTH_SIZE; j++) {
				complex left_aux, right_aux;
				float g = ((float) j) / ((float) (AZIMUTH_SIZE - 1));
				left_aux2 = left_channel[i];
				right_aux2 = right_channel[i];
				complex_mul_scalar(&left_aux2, g);
				complex_mul_scalar(&right_aux2, g);
				complex_sub(&right_aux, &left_channel[i], &right_aux2);
				complex_sub(&left_aux, &right_channel[i], &left_aux2);
				left_freq_azimuth_plane[i][j] = complex_abs(&left_aux);
				right_freq_azimuth_plane[i][j] = complex_abs(&right_aux);
			}
		}
		/* resynthesis: calulate the resynthesis peaks */
		for (i = 0; i < BUFFER_SIZE; i++) {
			left_resynthesis_peaks[i] = 0;
			right_resynthesis_peaks[i] = 0;
			for (j = AZIMUTH_POS_FROM; j <= AZIMUTH_POS_TO; j++) {
				left_resynthesis_peaks[i] += left_freq_azimuth_plane[i][j];
				right_resynthesis_peaks[i] += right_freq_azimuth_plane[i][j];
			}
		}
		/* resynthesis: calculate the final output spectrum and do the IFFT */
		for (i = 0; i < BUFFER_SIZE; i++) {
			float left_angle = (float) atan2(left_channel[i].im, left_channel[i].re);
			float right_angle = (float) atan2(right_channel[i].im, right_channel[i].re);
			left_channel_resynthesis[i].re = left_resynthesis_peaks[i] * cos(left_angle);
			left_channel_resynthesis[i].im = left_resynthesis_peaks[i] * sin(left_angle);
			right_channel_resynthesis[i].re = right_resynthesis_peaks[i] * cos(right_angle);
			right_channel_resynthesis[i].im = right_resynthesis_peaks[i] * sin(right_angle);
		}
		ifft(left_channel_resynthesis, L2_BUFFER_SIZE);
		ifft(right_channel_resynthesis, L2_BUFFER_SIZE);
		/* write the output */
		/* write in stereo format so we can re-apply the method over a previous output */
		for (i = 0; i < BUFFER_SIZE; i++) {
			left_output[i] = left_channel_resynthesis[i].re;
			right_output[i] = right_channel_resynthesis[i].re;
		}
		buffer_play(1, left_output, right_output, hanning_window);
	}
	return 0;
}
