
/*
	Emulador de lnea de bajo de tipo TB303 de Roland

	(c) Avelino Herrera Morales

	This software is free software; you can redistribute it and/or modify
	it under the terms of the GNU  General Public License as
	published by the Free Software Foundation; either version 2 of
	the License, or (at your option) any later version.
																																																																				   
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU  General Public License for more details.
																																																																				   
	You should have received a copy of the GNU  General Public
	License along with this software; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
	02111-1307, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <unistd.h>
#include "audio.h"

/* frecuencia de muestreo */
#define  SAMPLE_RATE  44100

/* notas musicales */
#define  DO       0
#define  DO_SUS   1
#define  RE       2
#define  RE_SUS   3
#define  MI       4
#define  FA       5
#define  FA_SUS   6
#define  SOL      7
#define  SOL_SUS  8
#define  LA       9
#define  LA_SUS   10
#define  SI       11

/* frecuencia del LA en la 4 octava del piano */
#define  LA4_FREQ             440
/* parmetros de la envolvente de la amplitud del oscilador */
#define  OSC_ATTACK           0.01
#define  OSC_DECAY            0.1
#define  OSC_ATTACK_SAMPLES   ((int)(OSC_ATTACK * SAMPLE_RATE))
#define  OSC_DECAY_SAMPLES    ((int)(OSC_DECAY * SAMPLE_RATE))
/* parmetros de la envolvente de la frecuencia de corte del filtro */
#define  FILT_ATTACK          0.02
#define  FILT_DECAY           0.1
#define  FILT_ATTACK_SAMPLES  ((int)(FILT_ATTACK * SAMPLE_RATE))
#define  FILT_DECAY_SAMPLES   ((int)(FILT_DECAY * SAMPLE_RATE))
/* para darle ese toque "acid" tambin hace falta un portamento :-) */
#define  PORTAMENTO_TIME      0.07
#define  PORTAMENTO_SAMPLES   ((int)(PORTAMENTO_TIME * SAMPLE_RATE))
/* duracin de una nota secuenciada */
#define  SEQ_TIME             0.11
#define  SAMPLES_PER_SEQ      ((int)(SEQ_TIME * SAMPLE_RATE))
/* nmero de notas que vamos a tocar */
#define  SEQ_SIZE             16

/* arrays con los valores a secuenciar de notas, octavas, amplitud de la
   envolvente del filtro, frecuencia de corte del filtro, resonancia del filtro
   y portamento:
   El programa recorre cclicamente estos 6 arrays y va ejecutando la meloda
   segn los parmetros de los mismos */
/* en cada posicin una nota (ver defines de arriba) o -1 para silencio */
int note_sequence[SEQ_SIZE] = {2, 1, 2, 3, 2, 2, 2, 2,
                               4, 1, 3, 3, 5, 8, 0, 2};
/* en cada posicin la octava de la nota del array anterior */
int octave_sequence[SEQ_SIZE] = {3, 3, 4, 2, 2, 3, 3, 2,
                                 3, 4, 2, 2, 3, 3, 3, 4};
/* en cada posicin la amplitud de la envolvente del filtro */
float famp_sequence[SEQ_SIZE] = {0.1, 0.9, 0.7, 0.5, 0.6, 0.7, 0.4, 0.5,
                                 0.1, 0.3, 0.5, 0.2, 0.3, 0.4, 0.1, 0.23};
/* en cada posicin la frecuencia de corte del filtro */
float cutoff_sequence[SEQ_SIZE] = {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
                                   0.0, 0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.2};
/* en cada posicin la resonancia del filtro */
float reso_sequence[SEQ_SIZE] = {0.1, 0.1, 0.1, 0.1, 0.2, 0.1, 0.1, 0.1,
                                 0.3, 0.8, 0.5, 0.2, 0.05, 0.02, 0.04, 0.09};
/* en cada posicin: 0=sin portamento, 1=con portamento */
int portamento_sequence[SEQ_SIZE] = {0, 1, 0, 1, 1, 0, 0, 0,
                                     0, 1, 1, 0, 1, 0, 0, 1};
/* variables relacionadas con el oscilador */
int osc_t;
int osc_max_t;             /* valor de osc_t en el que se alcanza el perodo de
                              la seal (1 / freq) */
float osc_ramp;            /* rampa para generar el diente de sierra */
float osc_old_freq = 0;    /* frecuencia de la nota anterior (portamento) */
float osc_freq = 20;       /* frecuencia de la nota actual */
float portamento_freq_inc; /* incremento de frecuencia en cada osc_t
                              (portamento) */
int do_portamento;         /* 0=nota actual sin portamento, 1=con portamento */
float pwm;                 /* un valor <0 hace que el oscilador genere un
                              diente de sierra, un valor entre 0 y 1 hace que
                              oscilador genere una onda cuadrada de diferentes
                              anchuras (ciclo de trabajo) */
int ad_t, fad_t;           /* "t"s para las envolventes de ampli y filtro */

float high = 0, mid = 0, low = 0;  /* valores del filtro de estado variable */

/* seal de salida */
float output[SAMPLES_PER_SEQ];                /* formato nativo */
signed short int to_device[SAMPLES_PER_SEQ];  /* para enviar a la tarjeta */

/******************************************************************************
	Devuelve la frecuencia en hercios de una nota (note=0..11) y en una octava
	determinada (octave).
	Un LA (note=9) en la 4 octava (octave=4) emite una frecuencia de 440Hz.
*/
float get_note_freq(int note, int octave) {
	return LA4_FREQ * pow(2.0, ((note - LA) / 12.0) + (octave - 4));
	}

/******************************************************************************
	Inicializa las variables para arrancar el oscilador, el filtro y las dos
	envolventes.
	freq       = Frecuencia de la nota en hercios.
	portamento = 0: para no generar portamento.
	             1: para generar portamento a partir de la nota anterior.
	dutty_cycle = ciclo de trabajo.   <0: onda de diente de sierra.
	                                0..1: onda cuadrada con diferentes ciclos
	                                      de trabajo.
*/
void osc_start(float freq, int portamento, float dutty_cycle) {
	/* inicializamos las "t" del oscilador */
	osc_t = 0;
	ad_t = 0;
	fad_t = 0;
	/* actualizamos las frecuencias y el pwm */
	osc_old_freq = osc_freq;
	osc_freq = freq;
	pwm = dutty_cycle;
	if (portamento) {
		/* si hay portamento, lo indicamos */
		do_portamento = 1;
		/* y calculamos el incremento frecuencial en cada muestra */
		portamento_freq_inc = (freq - osc_old_freq) / PORTAMENTO_SAMPLES;
		}
	 else {
		/* si no hay portamento, lo indicamos */
		do_portamento = 0;
		/* y calculamos la rampa del diente de sierra */
		osc_max_t = (int) rint(SAMPLE_RATE / freq);
		osc_max_t--;
		osc_ramp = 2.0 / osc_max_t;
		}
	return;
	}

/******************************************************************************
	Genera n muestras del oscilador sobre el buffer out.
	cut       = frecuencia de corte del filtro.
	reso      = resonancia del filtro.
	famp_mult = amplitud de la envolvente del filtro.
*/
void osc_work(float *out, int n, float cut, float reso, float famp_mult) {
	int i;
	float amp, famp, temp_cut;

	for (i = 0; i < n; i++) {
		/* calculamos la envolvente de amplitud */
		if (ad_t < OSC_ATTACK_SAMPLES)
			amp = (float)ad_t / OSC_ATTACK_SAMPLES;
		else if ((ad_t - OSC_ATTACK_SAMPLES) < OSC_DECAY_SAMPLES)
			amp = 1.0 - (float)(ad_t - OSC_ATTACK_SAMPLES) / OSC_DECAY_SAMPLES;
		else
			amp = 0.0;
		/* miramos aver si hay portamento en esta nota */
		if (do_portamento) {
			/* si hay portamento, vamos incrementando la frecuencia de la nota
			   anterior */
			osc_old_freq += portamento_freq_inc;
			/* hasta que alcance la frecuencia de la nota actual */
			if (osc_old_freq > osc_freq) {
				osc_old_freq = osc_freq;
				do_portamento = 0;
				}
			/* calculamos la rampa de la seal de diente de sierra */
			osc_max_t = (int) rint(SAMPLE_RATE / osc_old_freq);
			osc_max_t--;
			osc_ramp = 2.0 / osc_max_t;
			}
		/* generamos la seal a partir de la rampa */
		out[i] = -1.0 + (osc_t * osc_ramp);
		/* si pwm>=0 generamos una onda cuadrada */
		if (pwm >= 0.0)
			out[i] = (out[i] > pwm) ? 1.0 : -1.0;
		/* aplicamos la envolvente de amplitud */
		out[i] *= amp;
		/* calculamos la envolvente del filtro */
		if (fad_t < FILT_ATTACK_SAMPLES)
			famp = (float)fad_t / FILT_ATTACK_SAMPLES;
		else if ((fad_t - FILT_ATTACK_SAMPLES) < FILT_DECAY_SAMPLES)
			famp = 1.0 - (float)(fad_t - FILT_ATTACK_SAMPLES) / FILT_DECAY_SAMPLES;
		else
			famp = 0.0;
		/* obtenemos la frecuencia de corte a partir de la frecuencia de corte
		   pasada por parmetro y la envolvente */
		temp_cut = cut + (famp_mult * famp);
		if (cut > 1.0)
			cut = 1.0;
		else if (cut < 0.0)
			cut = 0.0;
		/* relizamos el filtrado mediante un filtro de estado variable */
		high = out[i] - (mid * reso) - low;
		mid += high * temp_cut;
		low += mid * temp_cut;
		/* la salida es la salida paso-bajo del filtro */
		out[i] = low;
		/* incrementamos las "t" */
		osc_t++;
		ad_t++;
		fad_t++;
		/* en cuanto osc_t llegue al perodo de la seal lo ponemos otra vez a
		   0 para que se vuelva a generar otra rampa (diente de sierra) */
		if (osc_t > osc_max_t)
			osc_t = 0;
		}
	return;
	}

/*****************************************************************************/
int main(void) {
	int i, j;
	int fd;
	float freq, cutoff;

	/* abrimos el dispositivo e audio */
	fd = audio_open(0);
	/* comenzamos en la secuencia 0 */
	i = 0;
	while (1) {
		if (note_sequence[i] < 0)
			/* si no hay nota emitimos un silencio */
			memset(output, 0, SAMPLES_PER_SEQ * sizeof(float));
		else {
			/* si hay nota, la convertimos en frecuencia */
			freq = get_note_freq(note_sequence[i], octave_sequence[i]);
			/* y la hacemos sonar */
			osc_start(freq, portamento_sequence[i], -1);
			osc_work(output, SAMPLES_PER_SEQ, cutoff_sequence[i],
			         reso_sequence[i], famp_sequence[i]);
			}
		/* convertimos las muestras float en muestras para el dispositivo */
		for (j = 0; j < SAMPLES_PER_SEQ; j++) {
			if (output[j] > 1.0)
				output[j] = 1.0;
			else if (output[j] < -1.0)
				output[j] = -1.0;
			to_device[j] = (signed short int) rint(output[j] * 32767);
			}
		/* escribimos en el dispositivo */
		write(fd, to_device, SAMPLES_PER_SEQ * sizeof(signed short int));
		/* a por la siguiente secuencia */
		i = (i + 1) % SEQ_SIZE;
		}
	return 0;
	}
