

#include <stdint.h>


using namespace std;


#define  NVIC_ISER    ((volatile uint32_t *) 0xE000E100)
#define  NVIC_ICER    ((volatile uint32_t *) 0xE000E180)
#define  NVIC_IPR     ((volatile uint32_t *) 0xE000E400)


#define  NVIC_ENABLE_IRQ(irq)   NVIC_ISER[(irq) >> 5] |= ((uint32_t) 1) << ((irq) & 0x0000001F)
#define  NVIC_DISABLE_IRQ(irq)  NVIC_ICER[(irq) >> 5] |= ((uint32_t) 1) << ((irq) & 0x0000001F)
#define  NVIC_SET_PRIORITY(irq, pri)   NVIC_IPR[(irq) >> 2] |= (NVIC_IPR[(irq) >> 2] & ~(0x000000FF << (((irq) & 3) << 3))) | (((uint32_t) (pri)) << (((irq) & 3) << 3))


#define  RCC_APB2ENR   *((volatile uint32_t *) 0x40021018)
#define  RCC_APB2RSTR  *((volatile uint32_t *) 0x4002100C)
#define  GPIOA_CRL     *((volatile uint32_t *) 0x40010800)
#define  GPIOA_CRH     *((volatile uint32_t *) 0x40010804)
#define  GPIOC_CRH     *((volatile uint32_t *) 0x40011004)
#define  GPIOC_ODR     *((volatile uint32_t *) 0x4001100C)
#define  USART_CR1     *((volatile uint32_t *) 0x4001380C)
#define  USART_BRR     *((volatile uint32_t *) 0x40013808)
#define  USART_DR      *((volatile uint32_t *) 0x40013804)
#define  USART_SR      *((volatile uint32_t *) 0x40013800)
#define  ADC_CR2       *((volatile uint32_t *) 0x40012408)
#define  ADC_DR        *((volatile uint32_t *) 0x4001244C)


#define  USART1_BUFFER_SIZE  4096


volatile uint8_t usart1Buffer[USART1_BUFFER_SIZE];
volatile uint16_t usart1BufferIn, usart1BufferOut;


void usart1Interrupt()  __attribute__ ((section(".usart1")));


void usart1Interrupt() {
	if (USART_SR & (((uint32_t) 1) << 6)) {
		// TX buffer empty
		if (usart1BufferIn != usart1BufferOut) {
			USART_DR = usart1Buffer[usart1BufferOut];
			usart1BufferOut++;
			if (usart1BufferOut == USART1_BUFFER_SIZE)
				usart1BufferOut = 0;
		}
		if (usart1BufferIn == usart1BufferOut)
			USART_CR1 &= ~(((uint32_t) 1) << 6);
	}
}


void usart1Init() {
	// usart1 enable, afio enable, gpioa enable
	RCC_APB2ENR |= (((uint32_t) 1) << 14) | ((uint32_t) 1) | (((uint32_t) 1) << 2);
	// PA9 (TX) pin is output, high speed, alternate function push-pull
	// PA10 (RX) pin is input, high speed, floating
	GPIOA_CRH = 0x444444B4;
	// enable usart
	USART_CR1 |= (((uint32_t) 1) << 13);
	// 72000000 / (16 * (468 + (12 / 16))) = 9600 bps
	USART_BRR = ((uint32_t) 12) | (((uint32_t) 468) << 4);
	// enable tx
	USART_CR1 |= (((uint32_t) 1) << 3);
	// wait tx complete for first dummy frame
	while (!(USART_SR & (((uint32_t) 1) << 6)))
		;
	usart1BufferIn = 0;
	usart1BufferOut = 0;
	NVIC_ENABLE_IRQ(37);
	NVIC_SET_PRIORITY(37, 255);   // lowest priority
}


void usart1SendByteBuffered(uint8_t b) {
	usart1Buffer[usart1BufferIn] = b;
	usart1BufferIn++;
	if (usart1BufferIn == USART1_BUFFER_SIZE)
		usart1BufferIn = 0;
}


void usart1Flush() {
	USART_CR1 |= (((uint32_t) 1) << 6);
}


void usart1SendBytes(uint8_t *p, uint32_t size) {
	while (size > 0) {
		usart1SendByteBuffered(*p);
		p++;
		size--;
	}
	usart1Flush();
}


void usart1SendString(const char *s) {
	while (*s != 0) {
		usart1SendByteBuffered((uint8_t) *s);
		s++;
	}
	usart1Flush();
}


void usart1SendHexBytes(uint8_t *p, uint32_t size, bool separateBytes = true, bool doFlush = true) {
	while (size > 0) {
		uint8_t b = (*p >> 4);
		usart1SendByteBuffered((b > 9) ? ('A' + (b - 10)) : ('0' + b));
		b = *p & 0x0F;
		usart1SendByteBuffered((b > 9) ? ('A' + (b - 10)) : ('0' + b));
		if (separateBytes)
			usart1SendByteBuffered(' ');
		p++;
		size--;
	}
	if (doFlush)
		usart1Flush();
}


void usart1SendHexValue(uint32_t v) {
	uint8_t x = (v >> 24) & 0xFF;
	usart1SendHexBytes(&x, 1, false, false);
	x = (v >> 16) & 0xFF;
	usart1SendHexBytes(&x, 1, false, false);
	x = (v >> 8) & 0xFF;
	usart1SendHexBytes(&x, 1, false, false);
	x = v & 0xFF;
	usart1SendHexBytes(&x, 1, false, false);
	usart1Flush();
}


#define  RCC_APB1RSTR  *((volatile uint32_t *) 0x40021010)
#define  RCC_APB1ENR   *((volatile uint32_t *) 0x4002101C)
#define  USB_CNTR      *((volatile uint16_t *) 0x40005C40)
#define  USB_ISTR      *((volatile uint16_t *) 0x40005C44)


void usbDeviceInit() {
	// enable USB clock
	RCC_APB1ENR |= (((uint32_t) 1) << 23);
	// enable USB interrupts
	NVIC_ENABLE_IRQ(20);
	NVIC_SET_PRIORITY(20, 0);   // highest priority
	// enable analog transceivers
	USB_CNTR &= ~(((uint16_t) 1) << 1);
	for (uint32_t i = 0; i < 20000; i++)
		;
	USB_CNTR &= ~(((uint16_t) 1) << 0);
	USB_ISTR = 0;
	// enable and wait for USB RESET interrupt
	USB_CNTR |= (((uint16_t) 1) << 10);
	usart1SendString("USB initialized\r\n");
}


#define  USB_BTABLE     *((volatile uint16_t *) 0x40005C50)
#define  USB_EP0R       *((volatile uint16_t *) 0x40005C00)
#define  USB_EP1R       *((volatile uint16_t *) 0x40005C04)
#define  USB_EP2R       *((volatile uint16_t *) 0x40005C08)
#define  USB_DADDR      *((volatile uint16_t *) 0x40005C4C)
#define  USB_ADDR0_TX   *((volatile uint16_t *) 0x40006000)
#define  USB_COUNT0_TX  *((volatile uint16_t *) 0x40006004)
#define  USB_ADDR0_RX   *((volatile uint16_t *) 0x40006008)
#define  USB_COUNT0_RX  *((volatile uint16_t *) 0x4000600C)
#define  USB_ADDR1_TX   *((volatile uint16_t *) 0x40006010)
#define  USB_COUNT1_TX  *((volatile uint16_t *) 0x40006014)
#define  USB_ADDR1_RX   *((volatile uint16_t *) 0x40006018)
#define  USB_COUNT1_RX  *((volatile uint16_t *) 0x4000601C)
#define  USB_ADDR2_TX   *((volatile uint16_t *) 0x40006020)
#define  USB_COUNT2_TX  *((volatile uint16_t *) 0x40006024)
#define  USB_ADDR2_RX   *((volatile uint16_t *) 0x40006028)
#define  USB_COUNT2_RX  *((volatile uint16_t *) 0x4000602C)
#define  USB_EP0TXBUF    ((volatile uint8_t *)  0x40006030)
#define  USB_EP0RXBUF    ((volatile uint8_t *)  0x40006040)
#define  USB_EP1TXBUF    ((volatile uint8_t *)  0x40006050)
#define  USB_EP2RXBUF    ((volatile uint8_t *)  0x40006060)


uint16_t usbRxBuffer[32];
uint16_t usbTxBuffer[32];


class UsbSetupPacket {
	public:
		uint8_t bmRequestType;
		uint8_t bRequest;
		uint16_t wValue;
		uint16_t wIndex;
		uint16_t wLength;
} __attribute__ ((packed));


struct UsbDeviceDescriptor {
	uint8_t bLength;
	uint8_t bDescriptorType;
	uint16_t bcdUSB;
	uint8_t bDeviceClass;
	uint8_t bDeviceSubClass;
	uint8_t bDeviceProtocol;
	uint8_t bMaxPacketSize;
	uint16_t idVendor;
	uint16_t idProduct;
	uint16_t bcdDevice;
	uint8_t iManufacturer;
	uint8_t iProduct;
	uint8_t iSerialNumber;
	uint8_t bNumConfigurations;
} __attribute__ ((packed));


const UsbDeviceDescriptor MyUsbDeviceDescriptor = {
	0x12,      // descriptor size
	0x01,      // descriptor type (device)
	0x0110,    // USB protocol version 1.10
	0x00,
	0x00,
	0x00,
	0x08,      // max packet size for control endpoint 0 = 8 bytes
	0xF055,    // vendor id
	0x0001,    // product id
	0x0100,
	0x00,
	0x00,
	0x00,
	0x01       // num configurations
};


struct UsbEndpointDescriptor {
	uint8_t bLength;
	uint8_t bDescriptorType;
	uint8_t bEndpointAddress;
	uint8_t bmAttributes;
	uint16_t wMaxPacketSize;
	uint8_t bInterval;
} __attribute__ ((packed));


struct UsbHIDDescriptor {
	uint8_t bLength;
	uint8_t bDescriptorType;
	uint16_t bcdHID;
	uint8_t bCountryCode;
	uint8_t bNumDescriptors;
	uint8_t bDescriptorType0;
	uint16_t wDescriptorLength0;
} __attribute__ ((packed));


struct UsbInterfaceDescriptor {
	uint8_t bLength;
	uint8_t bDescriptorType;
	uint8_t bInterfaceNumber;
	uint8_t bAlternateSetting;
	uint8_t bNumEndpoints;
	uint8_t bInterfaceClass;
	uint8_t bInterfaceSubClass;
	uint8_t bInterfaceProtocol;
	uint8_t iInterface;
} __attribute__ ((packed));


struct UsbConfigurationDescriptor {
	uint8_t bLength;
	uint8_t bDescriptorType;
	uint16_t wTotalLength;
	uint8_t bNumInterfaces;
	uint8_t bConfigurationValue;
	uint8_t iConfiguration;
	uint8_t bmAttributes;
	uint8_t bMaxPower;
	UsbInterfaceDescriptor interface;
	UsbEndpointDescriptor inEndpoint;
	UsbEndpointDescriptor outEndpoint;
} __attribute__ ((packed));


const UsbConfigurationDescriptor MyUsbConfigurationDescriptor = {
	0x09,    // descriptor size
	0x02,    // descriptor type (configuration)
	0x0020,  // configuration (9) + interface (9) + endpoint (7) + endpoint (7) = 32
	0x01,    // num interfaces = 1
	0x01,    // this configuration number = 1
	0x00,
	0x80,    // bus powered (not self powered)
	0x20,    // 32 * 2 = 64 mA
	{           // interface descriptor
		0x09,   // descriptor size
		0x04,   // descriptor type (interface)
		0x00,   // interface number (zero based)
		0x00,
		0x02,   // num endpoints = 2
		0xFF,   // class = vendor defined
		0xFF,   // subclass = vendor defined
		0x00,
		0x00
	},
	{             // in endpoint descriptor
		0x07,     // descriptor size
		0x05,     // descriptor type (endpoint)
		0x81,     // in endpoint 1
		0x02,     // bulk endpoint
		0x0008,   // max packet size = 8 bytes
		0x0A      // 10 ms for polling interval
	},
	{             // out endpoint descriptor
		0x07,     // descriptor size
		0x05,     // descriptor type (endpoint)
		0x02,     // out endpoint 2
		0x02,     // bulk endpoint
		0x0008,   // max packet size = 8 bytes
		0x0A      // 10 ms for polling interval
	}
};


struct UsbString0Descriptor {
	uint8_t bLength;
	uint8_t bDescriptorType;
	uint16_t wLangId0;
} __attribute__ ((packed));


const UsbString0Descriptor MyUsbString0Descriptor = {
	0x04,            // descriptor size
	0x03,            // descriptor type (string descriptor)
	0x0409           // 'en_US' language id
};


const uint16_t MyUsbStatus = 0x0000;


void usbCopyFromPacketSRAM(volatile uint32_t *packetSRAMSource, volatile uint16_t *destination, uint16_t bytes) {
	volatile uint32_t *p = (volatile uint32_t *) packetSRAMSource;
	volatile uint16_t *q = destination;
	uint16_t n = bytes >> 1;
	if (bytes & 1)
		n++;
	for (uint16_t i = 0; i < n; i++, p++, q++)
		*q = (uint16_t) (*p & 0x0000FFFF);
}


void usbCopyToPacketSRAM(volatile uint16_t *source, volatile uint32_t *packetSRAMDestination, uint16_t bytes) {
	volatile uint32_t *p = (volatile uint32_t *) packetSRAMDestination;
	volatile uint16_t *q = source;
	uint16_t n = bytes >> 1;
	if (bytes & 1)
		n++;
	for (uint16_t i = 0; i < n; i++, p++, q++)
		*p = (uint32_t) *q;
}


volatile uint16_t usbNextAddress;


void usbDeviceISR()  __attribute__ ((section(".usblp")));


#define  STAT_RX_DISABLE  0x0000
#define  STAT_RX_STALL    0x1000
#define  STAT_RX_VALID    0x3000
#define  STAT_RX_MASK     0x3000
#define  STAT_TX_DISABLE  0x0000
#define  STAT_TX_STALL    0x0010
#define  STAT_TX_VALID    0x0030
#define  STAT_TX_MASK     0x0030
#define  DTOG_RX_MASK     0x4000
#define  DTOG_TX_MASK     0x0040


volatile uint16_t ep0DataCount;
volatile uint8_t *ep0DataPtr;


#define  TOGGLE_SET_0(val, bit)  ((val) & (((uint16_t) 1) << (bit)))
#define  TOGGLE_SET_1(val, bit)  (((val) & (((uint16_t) 1) << (bit))) ^ (((uint16_t) 1) << (bit)))


#define  STAT_RX_DIS(val)  (TOGGLE_SET_0(val, 13) | TOGGLE_SET_0(val, 12))
#define  STAT_RX_STA(val)  (TOGGLE_SET_0(val, 13) | TOGGLE_SET_1(val, 12))
#define  STAT_RX_NAK(val)  (TOGGLE_SET_1(val, 13) | TOGGLE_SET_0(val, 12))
#define  STAT_RX_VAL(val)  (TOGGLE_SET_1(val, 13) | TOGGLE_SET_1(val, 12))


#define  STAT_TX_DIS(val)  (TOGGLE_SET_0(val, 5) | TOGGLE_SET_0(val, 4))
#define  STAT_TX_STA(val)  (TOGGLE_SET_0(val, 5) | TOGGLE_SET_1(val, 4))
#define  STAT_TX_NAK(val)  (TOGGLE_SET_1(val, 5) | TOGGLE_SET_0(val, 4))
#define  STAT_TX_VAL(val)  (TOGGLE_SET_1(val, 5) | TOGGLE_SET_1(val, 4))


#define  DTOG_RX_0(val)  TOGGLE_SET_0(val, 14)
#define  DTOG_RX_1(val)  TOGGLE_SET_1(val, 14)
#define  DTOG_TX_0(val)  TOGGLE_SET_0(val, 6)
#define  DTOG_TX_1(val)  TOGGLE_SET_1(val, 6)


#define  STAT_VAL  3
#define  STAT_NAK  2
#define  STAT_STA  1


void usbDeviceEPRSetRxStat(volatile uint16_t &epr, uint16_t stat) {
    uint16_t val, tog;
    val = epr & 0x8F8F;
    val |= (epr & 0x3000) ^ (stat << 12);
	epr = val;
}


void usbDeviceEPRSetTxStat(volatile uint16_t &epr, uint16_t stat) {
	uint16_t val = epr & 0x8F8F;
    val |= (epr & 0x0030) ^ (stat << 4);
	epr = val;
}


void usbDeviceEPRSetDtogRx(volatile uint16_t &epr, uint16_t dtog) {
	uint16_t val = epr & 0x8F8F;
	val |= (epr & 0x4000) ^ (dtog << 14);
	epr = val;
}


void usbDeviceEPRSetDtogTx(volatile uint16_t &epr, uint16_t dtog) {
	uint16_t val = epr & 0x8F8F;
	val |= (epr & 0x0040) ^ (dtog << 6);
	epr = val;
}


void usbDeviceISRReset() {
	// prepare buffer descriptor table for endpoint 0 (control)
	USB_BTABLE = 0;
	// endpoint 0 (bidireccional)
	USB_ADDR0_TX = 24;
	USB_COUNT0_TX = 0;   // 8
	USB_ADDR0_RX = 32;
	USB_COUNT0_RX = (((uint16_t) 4) << 10);   // 2 * 4 = 8 bytes
	// endpoint 1 (in, tx)
	USB_ADDR1_TX = 40;
	USB_COUNT1_TX = 0;   // 8
	USB_ADDR1_RX = 40;
	USB_COUNT1_RX = (((uint16_t) 4) << 10);   // 2 * 4 = 8 bytes
	// endpoint 2 (out, rx)
	USB_ADDR2_TX = 48;
	USB_COUNT2_TX = 0;   // 8
	USB_ADDR2_RX = 48;
	USB_COUNT2_RX = (((uint16_t) 4) << 10);   // 2 * 4 = 8 bytes
	// device address = 0
	USB_DADDR = ((uint16_t) 1) << 7;   // enable usb function
	usbNextAddress = 0;
	// prepare endpoint 0 for rx setup packets
	USB_EP0R = (((uint16_t) 1) << 9);
	usbDeviceEPRSetRxStat(USB_EP0R, STAT_NAK);
	usbDeviceEPRSetTxStat(USB_EP0R, STAT_NAK);
	usbDeviceEPRSetDtogRx(USB_EP0R, 0);
	usbDeviceEPRSetDtogTx(USB_EP0R, 0);
	// prepare endpoint 1 and endpoint 2
	USB_EP1R = 1;
	usbDeviceEPRSetRxStat(USB_EP1R, STAT_NAK);
	usbDeviceEPRSetTxStat(USB_EP1R, STAT_VAL);
	usbDeviceEPRSetDtogRx(USB_EP1R, 0);
	usbDeviceEPRSetDtogTx(USB_EP1R, 0);
	USB_EP2R = 2;
	usbDeviceEPRSetRxStat(USB_EP2R, STAT_VAL);
	usbDeviceEPRSetTxStat(USB_EP2R, STAT_NAK);
	usbDeviceEPRSetDtogRx(USB_EP2R, 0);
	usbDeviceEPRSetDtogTx(USB_EP2R, 0);
	// enable complete transfer interrupt
	USB_CNTR |= (((uint16_t) 1) << 15);
}


void usbDeviceISR() {
	uint16_t istr = USB_ISTR;
	if (istr & (((uint16_t) 1) << 10)) {
		//usart1SendString("USB reset\r\n");
		usbDeviceISRReset();
		USB_ISTR = 0;
	}
	else if (istr & (((uint16_t) 1) << 15)) {   // correct transfer interrupt
		USB_ISTR = 0;
		uint16_t epNum = istr & 0x000F;
		if (epNum == 0) {
			if (istr & 0x0010) {
				// out/setup packet
				if (USB_EP0R & (((uint16_t) 1) << 11)) {
					// setup packet
					usart1SendString("setup\r\n");
					usbCopyFromPacketSRAM((uint32_t *) USB_EP0RXBUF, usbRxBuffer, USB_COUNT0_RX & 0x03FF);
					UsbSetupPacket *setupPacket = (UsbSetupPacket *) usbRxBuffer;
					if ((setupPacket->bmRequestType == 0x80) && (setupPacket->bRequest == 0x06)) {
						bool stall = false;
						if ((setupPacket->wValue >> 8) == 1) {
							usart1SendString("\tgdev ");
							usart1SendHexValue(setupPacket->wLength);
							usart1SendString("\r\n");
							ep0DataPtr = (uint8_t *) &MyUsbDeviceDescriptor;           // get_descriptor (device)
							ep0DataCount = (sizeof(MyUsbDeviceDescriptor) < setupPacket->wLength) ? sizeof(MyUsbDeviceDescriptor) : setupPacket->wLength;
						}
						else if ((setupPacket->wValue >> 8) == 2) {
							usart1SendString("\tgconf ");
							usart1SendHexValue(setupPacket->wLength);
							usart1SendString("\r\n");
							ep0DataPtr = (uint8_t *) &MyUsbConfigurationDescriptor;    // get_descriptor (configuration)
							ep0DataCount = (sizeof(MyUsbConfigurationDescriptor) < setupPacket->wLength) ? sizeof(MyUsbConfigurationDescriptor) : setupPacket->wLength;
						}
						else if ((setupPacket->wValue >> 8) == 3) {
							usart1SendString("\tgstr ");
							usart1SendHexValue(setupPacket->wLength);
							usart1SendString("\r\n");
							ep0DataPtr = (uint8_t *) &MyUsbString0Descriptor;    // get_descriptor (string)
							ep0DataCount = (sizeof(MyUsbString0Descriptor) < setupPacket->wLength) ? sizeof(MyUsbString0Descriptor) : setupPacket->wLength;
						}
						else {
							usart1SendString("\tg?");
							usart1SendHexValue(setupPacket->wValue >> 8);
							usart1SendString("\r\n");
							ep0DataCount = 0;
							stall = true;
						}
						if (stall)
							usbDeviceEPRSetTxStat(USB_EP0R, STAT_STA);
						else {
							uint16_t size = (ep0DataCount > 8) ? 8 : ep0DataCount;     // copy bytes to packet SRAM
							usbCopyToPacketSRAM((uint16_t *) ep0DataPtr, (uint32_t *) USB_EP0TXBUF, size);
							USB_COUNT0_TX = size;
							ep0DataCount -= size;
							ep0DataPtr += size;
							usbDeviceEPRSetTxStat(USB_EP0R, STAT_VAL);
						}
						usbDeviceEPRSetRxStat(USB_EP0R, STAT_STA);
					}
					else if ((setupPacket->bmRequestType == 0x00) && (setupPacket->bRequest == 0x05)) {
						usart1SendString("\tsaddr\r\n");
						usbNextAddress = setupPacket->wValue;
						USB_COUNT0_TX = 0;
						usbDeviceEPRSetTxStat(USB_EP0R, STAT_VAL);
						usbDeviceEPRSetRxStat(USB_EP0R, STAT_STA);
					}
					else if ((setupPacket->bmRequestType == 0x00) && (setupPacket->bRequest == 0x09)) {
						usart1SendString("\tsconf\r\n");
						USB_COUNT0_TX = 0;
						usbDeviceEPRSetTxStat(USB_EP0R, STAT_VAL);
						usbDeviceEPRSetRxStat(USB_EP0R, STAT_STA);
					}
					else if ((setupPacket->bmRequestType == 0x80) && (setupPacket->bRequest == 0x00)) {
						usart1SendString("\tgstat ");
						usart1SendHexValue(setupPacket->wLength);
						usart1SendString("\r\n");
						usbCopyToPacketSRAM((uint16_t *) &MyUsbStatus, (uint32_t *) USB_EP0TXBUF, 2);
						USB_COUNT0_TX = 2;
						usbDeviceEPRSetTxStat(USB_EP0R, STAT_VAL);
						usbDeviceEPRSetRxStat(USB_EP0R, STAT_STA);
					}
					else {
						usart1SendString("\tother setup packet\r\n");
						usart1SendString("x: ");
						usart1SendHexValue(setupPacket->bmRequestType);
						usart1SendString(" ");
						usart1SendHexValue(setupPacket->bRequest);
						usart1SendString("\r\n");
					}
				}
				else {
					// out packet
					usart1SendString("o\r\n");
					usbCopyFromPacketSRAM((uint32_t *) USB_EP0RXBUF, usbRxBuffer, USB_COUNT0_RX & 0x03FF);
					// TODO process data
				}
			}
			else {
				// in packet
				usart1SendString("i\r\n");
				if (usbNextAddress != 0) {
					USB_DADDR = (((uint16_t) 1) << 7) | (usbNextAddress & 0x007F);
					usbNextAddress = 0;
				}
				else {
					uint16_t size = (ep0DataCount > 8) ? 8 : ep0DataCount;
					usbCopyToPacketSRAM((uint16_t *) ep0DataPtr, (uint32_t *) USB_EP0TXBUF, size);
					USB_COUNT0_TX = size;
					ep0DataCount -= size;
					ep0DataPtr += size;
					usbDeviceEPRSetTxStat(USB_EP0R, STAT_VAL);
					usbDeviceEPRSetRxStat(USB_EP0R, STAT_VAL);
				}
			}
			USB_EP0R &= 0x0F0F;    // ctr_rx = 0, ctr_tx = 0
		}
		else if (epNum == 1) {
			if (istr & 0x0010) {
				// out packet
			}
			else {
				// in packet 
			}
			usbDeviceEPRSetTxStat(USB_EP1R, STAT_VAL);
			usbDeviceEPRSetRxStat(USB_EP1R, STAT_NAK);
			USB_EP1R &= 0x0F0F;    // ctr_rx = 0, ctr_tx = 0
		}
		else if (epNum == 2) {
			if (istr & 0x0010) {
				// out packet
				usbCopyFromPacketSRAM((uint32_t *) USB_EP2RXBUF, usbRxBuffer, USB_COUNT2_RX & 0x03FF);
				usart1SendString("rx '");
				usart1SendBytes((uint8_t *) usbRxBuffer, USB_COUNT2_RX & 0x03FF);
				usart1SendString("'\r\n");
			}
			else {
				// in packet 
			}
			usbDeviceEPRSetTxStat(USB_EP2R, STAT_NAK);
			usbDeviceEPRSetRxStat(USB_EP2R, STAT_VAL);
			USB_EP2R &= 0x0F0F;    // ctr_rx = 0, ctr_tx = 0
		}
		//USB_ISTR = 0;
	}
}


int main() {
	// enable clock on port C
	RCC_APB2ENR |= ((uint32_t) 1) << 4;
	// PC13 pin is output, low speed, push-pull
	GPIOC_CRH = 0x44244444;
	// force PC13 pin to 1 (led off) at startup
	GPIOC_ODR |= (((uint32_t) 1) << 13);
	// enable USART TX for debug
	usart1Init();
	usart1SendString("stm32f10x-usbdevice\r\n");
	usart1SendString("(c) 2017 Avelino Herrera Morales\r\n");
	// enable USB device
	usbDeviceInit();
	while (true) {
		// WFI (wait for interrupt) instruction, enters low power mode
		asm volatile ("wfi");
	}
}
